function [ ] = LUIM0_map_setup( )
%
% Land Use Intensity Module - function 0 (LUIM0)
%
% This file loads and formats the raw raster maps that are needed for the
% intensity module over the maps that are used in the allocation module.
% The vectorized maps are saved in excel and matlab-data formats.
%
% This function is not run during the execution of the intensity module
% (or LURNZ). It is run once to setup the maps that LURNZ takes as input.
% If and only if the raw / underlying data is updated should this script be
% rerun.
%

% Code by Simon Anastasiadis : 2012-01-13

%% Parameters

% Resolution
% This depends on the resolution of the underlying maps, currently only two
% resolutions are available: 25ha per pixel and 1ha per pixel.
resolution = 25;
% resolution = 1; % CURRENTLY NOT READY!

% output path
output_path = '..\Data for Modules\Data 3 Land Use Intensity Module\';

% number of maps
num_maps = 6;

%% Setup path

if resolution == 25
    load_folder = 'Raster ASCI - 25 ha';
elseif resolution == 1
    load_folder = 'Raster ASCI - 1 ha';
end

% shared load path
map_path = '..\..\..\Data\';
% specific load path
map_path = [map_path, load_folder];

%% Set map names & paths

% base map / starting map
base_map = [map_path,'\','LandUseMap_Year_2008.asc'];

% CCAV map
ccav_map = [map_path,'\','ccav_asc.asc'];
% Sheep/beef farm class map
SBclass_map = [map_path,'\','newSBClassRegionMap.asc'];
% LIC map
LIC_map = [map_path,'\','ta_asc.asc'];
% Regional council
RC_map = [map_path,'\','rc_asc.asc'];

% We construct LIC regions by merging adjacent TAs

%% Load base_map into Matlab

% inform user
msg = sprintf(' - - LUIM0 : loading base map');
disp(msg)

% loadASCI map files
% map = importdata( PATH , DELIMITER = SPACE , HEADER_LINES = 6 )
base_map  = importdata(base_map, ' ', 6);

% inform user
msg = sprintf(' - - LUIM0 : base map loaded');
disp(msg)

%% Construct GIS header information

GISheader = ConstructGISHeader(base_map.textdata, output_path, resolution );

%% Load, Combine and vectorize all maps - setup
% we process and clear each map one at a time to save space
% (at 1 ha resolution loading 10 maps fills >90% of available memory and
% Matlab does not have enough space to process the maps).

% inform user
msg = sprintf(' - - LUIM0 : loading & combining all other maps');
disp(msg)

% establish GISINTENSITYDATA
ncol = GISheader(1);
nrow = GISheader(2);
GISINTENSITY = zeros( nrow * ncol ,num_maps+1);

% column tracker
c.sorter = 1;
c.base_map = 2;
c.ccav = 3;
c.sbclass = 4;
c.lic = 5;
c.rc = 6;
c.ta = 7;

% fill sorter and base map column
GISINTENSITY(:,c.sorter)   = (1:nrow*ncol).';
GISINTENSITY(:,c.base_map) = reshape(base_map.data,  nrow*ncol, 1);

% indicator for land pixels
sea_pixel = GISheader(6);
iNonSea = GISINTENSITY(:,c.base_map) ~= sea_pixel;

% trim sorter and base map in LURNZ DATA
GISINTENSITY = GISINTENSITY(iNonSea,:);

%% Load, Combine and vectorize all maps - map work
% for each map:
% - load map
% - use iNonSea to extract land pixels
% - save land pixels to LURNZDATA
% - clear map to save space

% note
% iNonSea and LURNZDATA(:,c.sorter) are both indicators of the land pixels
% map.data(iNonSea) and map.data(LURNZDATA(:,c.sorter))
% are equivalent as both extract from map.data the land pixels
% we use iNonSea in this code

% map = importdata( PATH , DELIMITER = SPACE , HEADER_LINES = 6 )
% LURNZDATA(:,c.map) = map.data(iNonSea);
% clear('map')

ccav_map   = importdata(ccav_map , ' ', 6);
ccav_map.data = CCAV_imputation(ccav_map.data, iNonSea);
GISINTENSITY(:,c.ccav)   = ccav_map.data(iNonSea);
clear('ccav_map')

SBclass_map = importdata(SBclass_map, ' ', 6);
GISINTENSITY(:,c.sbclass) = SBclass_map.data(iNonSea);
clear('SBclass_map')

LIC_map = importdata(LIC_map, ' ', 6);
GISINTENSITY(:,c.lic) = LIC_map.data(iNonSea);
GISINTENSITY(:,c.ta)  = LIC_map.data(iNonSea);
clear('LIC_map')

RC_map =  importdata(RC_map, ' ', 6);
GISINTENSITY(:,c.rc) = RC_map.data(iNonSea);
clear('RC_map')

% inform user
msg = sprintf(' - - LUIM0 : maps combined');
disp(msg)

%% Clean maps

% CCAV map is multiplied by 10 (to preserve decimal points in asci format)
% Reverse this multiplication
iCCAV = GISINTENSITY(:,c.ccav) == -9999;
GISINTENSITY(~iCCAV,c.ccav)   = GISINTENSITY(~iCCAV,c.ccav) / 10;

% Pixels with zero / undefined sheep beef class have national average class
% (this is value 9)
GISINTENSITY( GISINTENSITY(:,c.sbclass) == 0, c.sbclass ) = 9;

% Sea pixels have CCAV = 0
GISINTENSITY( GISINTENSITY(:,c.ccav) == sea_pixel, c.ccav ) = 0;
% Sea pixels have Sheep/beef class zero
GISINTENSITY( GISINTENSITY(:,c.sbclass) == sea_pixel, c.sbclass ) = 0;

% This appears to be all the cleaning necessary at 25ha resolution.

% remove iNonSea to save memory space
clear('iNonSea')

%% Split South Island Sheep Beef classes
% The Sheep beef class map combines West Coast, Nelson and Tasman with
% Canterbury and Marlborough.
% We wish to treat these separately.

% West Coast, Nelson and Tasman are made up perfectly by TAs with the IDs: 
% 51, 52, 55, 56, 57

% indicator
iSplitSI = GISINTENSITY(:,c.lic) == 51 | ...
           GISINTENSITY(:,c.lic) == 52 | ...
           GISINTENSITY(:,c.lic) == 55 | ...
           GISINTENSITY(:,c.lic) == 56 | ...
           GISINTENSITY(:,c.lic) == 57;
% We convert scores of 41, 42, 46 and 48 to 1, 2, 6 and 8 respectively
% Take modulus
GISINTENSITY(iSplitSI,c.sbclass) = mod(GISINTENSITY(iSplitSI,c.sbclass),10);

%% Convert TAs to LIC regions
% LIC regions are constructed from merging adjasent TAs

GISINTENSITY(:,c.lic) = ConvertTAtoLIC(GISINTENSITY(:,c.lic));

%% Remove unneeded columns

% Remove coefficient inputs columns and uTotal
GISINTENSITY(:,c.base_map) = [];

% set new column header
clear('c')
c.sorter = 1;
c.ccav = 2;
c.sbclass = 3;
c.lic = 4;
c.rc = 5;
c.ta = 6;

%% Save LURNZ GIS DATA information to excel file
% note that LUIM loads LURNZ GIS DATA from the Matlab data file
% But we used to save as an excel file too for historic reasons. 
% 
% % we use dlmwrite as csvwrite is less accurate
% % e.g. 1234567 is recorded as 1234500

%%  Save Matlab output

% remove unwanted files
clear('column_names', 'file_name', 'map_path', 'base_map', 'iSplitSI', ...
      'ncol', 'nrow', 'num_maps', 'load_folder', 'sea_pixel', 'msg');
%

% save output
if resolution==25
    save([output_path,'LUIM_input_data_',num2str(resolution),'ha'])
elseif resolution==1
    save([output_path,'LUIM_input_data_',num2str(resolution),'ha'],'-v7.3')
end

%% Inform User saving land use change is complete

msg = sprintf(' - - LUIM0 : setup data complete');
disp(msg)

end

% Subfunction : Construct GIS Header info from ASCI files

function GISheader = ConstructGISHeader(textdata, output_path, resolution )

% Map_GISHeader = [Longitude;Latitude;Xll;Yll;CellSize;No_Value]
% We construct the GISheader from map.textdata
% map.text data should be identical for all maps by construction

Longitude = sscanf(char(textdata(1)),'%*s %f');
Latitude  = sscanf(char(textdata(2)),'%*s %f');
Xll = sscanf(char(textdata(3)),'%*s %f');
Yll = sscanf(char(textdata(4)),'%*s %f');
CellSize  = sscanf(char(textdata(5)),'%*s %f');
No_Value  = sscanf(char(textdata(6)),'%*s %f');

% map.textdata contains the GISheader info
% char converts it from a cell array to a character string
% sscanf breaks the string into its component pieces

GISheader = [Longitude; Latitude; Xll; Yll; CellSize; No_Value];

%% Save GIS header information to excel file
% note that LUAM loads GISheader from the Matlab data file

% file and sheet name
file_name = ['GIS information - ',num2str(resolution),'ha.xls'];
sheet_name = 'Sheet1';

% header labels
GIS_labels = {'ncols'; 'nrows'; 'xllcorner'; 'yllcorner'; ...
              'cellsize'; 'NODATA_value'};

% write to Excel
xlswrite([output_path,file_name],GIS_labels,sheet_name,'A1');
xlswrite([output_path,file_name],GISheader ,sheet_name,'B1');

end

% Subfunction : Convert TA codes to LIC codes via concordance
function LIC = ConvertTAtoLIC(TA)

% construct empty vector
LIC = zeros(size(TA));

% Northland
LIC(TA == 1 | TA == 2 | TA == 3) = 1;
% Central Auckland
LIC(TA == 4 | TA == 5 | TA == 6 | TA == 7 | TA == 8 | TA == 9 | TA == 10) = 2;
% South Auckland
LIC(TA == 11 | TA == 12 | TA == 13 | TA == 15 | TA == 16 | TA == 17 | TA == 18 | TA == 19) = 3;
% Bay of Plenty
LIC(TA == 22 | TA == 23 | TA == 25 | TA == 26 | TA == 27) = 4;
% Central Plateau
LIC(TA == 21 | TA == 24) = 5;
% Western Uplands
LIC(TA == 20 | TA == 36) = 6;
% East Coast
LIC(TA == 28 | TA == 29) = 7;
% Hawkes Bay
LIC(TA == 30 | TA == 31 | TA == 32) = 8;
% Taranaki
LIC(TA == 33 | TA == 34 | TA == 35) = 9;
% Wellington
LIC(TA == 37 | TA == 38 | TA == 39 | TA == 40 | TA == 42 | TA == 43 | TA == 44 | TA == 45 | TA == 46 | TA == 47) = 10;
% Wairarapa
LIC(TA == 41 | TA == 48 | TA == 49 | TA == 50) = 11;
% Nelson / Marlborough
LIC(TA == 51 | TA == 52 | TA == 53 | TA == 54) = 12;
% West Coast
LIC(TA == 55 | TA == 56 | TA == 57) = 13;
% North Canterbury
LIC(TA == 58 | TA == 59 | TA == 60 | TA == 61 | TA == 62 | TA == 63) = 14;
% South Canterbury
LIC(TA == 64 | TA == 65 | TA == 66) = 15;
% Otago
LIC(TA == 68 | TA == 69 | TA == 70 | TA == 71 | TA == 72) = 16;
% Southland
LIC(TA == 73 | TA == 74 | TA == 75) = 17;

end

% Subfunction : Impute missing CCAV values
function ccav_map = CCAV_imputation(ccav_map, iNonSea)
% Some pixels do not have a CCAV value
% We impute these using nearest neighbour assignment
% (mostly coast, rivers, and islands)

% sizes
nrows = size(ccav_map,1);
ncols = size(ccav_map,2);

% iNonSea is passed to thi subfunction as a vector. reshpae into grid
iNonSea = reshape(iNonSea,nrows,ncols);

% run this code 3 times (to impute values for pixels up to three pixels
% away from a pixel with a ccav value)
for ii = 1:3
    
    % construct nearest neighbour map - northward neighbour
    neighbour_map = [ccav_map(2:end,:) ; -9999 * ones(1,ncols)];
    % identify pixels with no ccav value that should have a ccav value
    iCCAV = ccav_map == -9999 & iNonSea & neighbour_map ~= -9999;
    % assign values
    ccav_map(iCCAV) = neighbour_map(iCCAV);

    % construct nearest neighbour map - southward neighbour
    neighbour_map = [-9999 * ones(1,ncols) ; ccav_map(1:end-1,:)];
    % identify pixels with no ccav value that should have a ccav value
    iCCAV = ccav_map == -9999 & iNonSea & neighbour_map ~= -9999;
    % assign values
    ccav_map(iCCAV) = neighbour_map(iCCAV);

    % construct nearest neighbour map - eastward neighbour
    neighbour_map = [ccav_map(:,2:end) , -9999 * ones(nrows,1)];
    % identify pixels with no ccav value that should have a ccav value
    iCCAV = ccav_map == -9999 & iNonSea & neighbour_map ~= -9999;
    % assign values
    ccav_map(iCCAV) = neighbour_map(iCCAV);

    % construct nearest neighbour map - westward neighbour
    neighbour_map = [-9999 * ones(nrows,1) , ccav_map(:,1:end-1)];
    % identify pixels with no ccav value that should have a ccav value
    iCCAV = ccav_map == -9999 & iNonSea & neighbour_map ~= -9999;
    % assign values
    ccav_map(iCCAV) = neighbour_map(iCCAV);
    
end

end

